/*:
 * @target MZ
 * @plugindesc Adds a script call to toggle the FPS meter display during gameplay with a larger number in the top-right corner.
 * @author Clay Davis
 *
 * @help
 * Use the script call `ToggleFPSMeter.toggle()` to activate or deactivate the FPS meter.
 */

(() => {
    class FPSMeter {
        constructor(options) {
            this.options = options || {};
            this.options.theme = this.options.theme || 'dark';
            this.options.decimals = this.options.decimals || 0;

            this.dom = null; // Initialize the DOM reference
            this.frames = 0;
            this.lastTime = performance.now();
            this.frameTime = 0;
            this.fps = 0;
            this.isAttached = false; // Track whether the meter is attached

            this.createDOM();
        }

        createDOM() {
            if (this.dom) return; // Avoid recreating the DOM

            this.dom = document.createElement('div');
            this.dom.style.position = 'absolute';
            this.dom.style.top = '10px';
            this.dom.style.right = '10px'; // Positioned in the top-right corner
            this.dom.style.pointerEvents = 'none';
            this.dom.style.zIndex = '9999';
            this.dom.style.fontSize = '24px'; // Larger font size
            this.dom.style.fontFamily = 'Arial, sans-serif';
            this.dom.style.backgroundColor = this.options.theme === 'dark' ? '#000' : '#fff';
            this.dom.style.color = this.options.theme === 'dark' ? '#00ff00' : '#000';
            this.dom.style.padding = '5px';
            this.dom.style.borderRadius = '5px';
            this.dom.style.display = 'none';

            // Append the FPS meter only if `document.body` is ready
            if (document.body) {
                document.body.appendChild(this.dom);
                this.isAttached = true;
            }
        }

        show() {
            this.createDOM(); // Ensure DOM exists
            if (this.dom && this.isAttached) {
                this.dom.style.display = 'block';
            }
        }

        hide() {
            if (this.dom && this.isAttached) {
                this.dom.style.display = 'none';
            }
        }

        toggle() {
            if (!this.dom || !this.isAttached) {
                this.createDOM();
                this.show();
            } else if (this.dom.style.display === 'none') {
                this.show();
            } else {
                this.hide();
            }
        }

        update() {
            if (!this.dom || !this.isAttached || this.dom.style.display === 'none') return;

            const now = performance.now();
            this.frameTime += (now - this.lastTime) / 1000;
            this.frames++;

            if (this.frameTime > 1) {
                this.fps = (this.frames / this.frameTime).toFixed(this.options.decimals);
                this.frames = 0;
                this.frameTime = 0;

                this.dom.textContent = `FPS: ${this.fps}`;
            }
            this.lastTime = now;
        }
    }

    // Ensure global FPSMeter is defined
    if (!window.FPSMeter) {
        window.FPSMeter = FPSMeter;
    }

    // ToggleFPSMeter namespace
    window.ToggleFPSMeter = window.ToggleFPSMeter || {};

    ToggleFPSMeter.toggle = function () {
        if (!Graphics._fpsMeter) {
            Graphics._fpsMeter = new FPSMeter({
                decimals: 0,
                theme: 'dark',
            });
            Graphics._fpsMeter.show();
        } else {
            Graphics._fpsMeter.toggle();
        }
    };

    // Update FPS meter with every game frame
    const _SceneManager_updateMain = SceneManager.updateMain;
    SceneManager.updateMain = function () {
        _SceneManager_updateMain.apply(this, arguments);
        if (Graphics._fpsMeter) {
            Graphics._fpsMeter.update();
        }
    };
})();